package simulator;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import service.Transition;
import service.ComposedTransition;
import service.ComposedState;
import service.DuplicateStateException;
import service.ServiceFactory;
import service.ServiceFactoryImpl;
import service.SimpleTransition;
import service.SimpleState;
import service.State;
import service.TransitionSystem;
import service.TransitionSystemImpl;

/**
 * 
 * @author daniele
 *
 */
public class SimulationStandardImpl extends Simulation {

	
	
	
	/*--------------------------AUXILIARY CLASSES----------------------*/
	
	
	/**
	 * Class that stores similarity relations between states:
	 * If S2 is similar to S1
	 * then S1 will be the target and S2 will be in the Set objective
	 *
	 * @param <S> First member State type
	 * @param <T> Second member State type
	 */
	private class Similar<S extends State, T extends State>{
		S target;
		HashSet<T> services;
	}
	
	/**
	 * Capsule for composed states
	 * Every composed state of the type {S0 S1 S2 S3}
	 * Is represented by capsule like this:
	 * CAPSULE{S0 CAPSULE{S1 CAPSULE{S2 CAPSULE{S3 null}}}}
	 * @author daniele
	 *
	 * @param <S> Type of the states to be encapsulated
	 */
	private class StateCapsule<S extends State>{
		S state;
		StateCapsule<S> sc;;
	}
	
	
	
	
	
	
	
	
	/*---------------------------INSTANCE VARIABLES-------------------------*/
	
	
	/**
	 * variable that stores the Target service transition system.
	 */
	private TransitionSystem<SimpleState, SimpleTransition> target;
	/**
	 * Stores the transition system derived by the simulation over the asynchronous product
	 */
	private TransitionSystem<ComposedState, ComposedTransition> composition;
	/**
	 * Stores the relations between target states and composed states (represents the real simulation relation)
	 */
	private HashSet<Similar<SimpleState, ComposedState>> relation; 
	/**
	 * Stores the asynchronous product of all the available services
	 */
	private TransitionSystem<ComposedState, ComposedTransition> async;
	
	
	
	
	/*----------------------------METHODS-----------------------------*/
	
	
	private SimulationStandardImpl(){}
	
	@Override
	/**
	 * Returns the transition system deriving by the simulation
	 */
	public TransitionSystem<ComposedState, ComposedTransition> getCompositionTs() {
		return composition;
	}

	@Override
	/**
	 * Returns the states (if any) similar to the targetState specified
	 */
	public Set<ComposedState> getSimilarStates(State targetState) {
		Similar<SimpleState, ComposedState> currentSimilar = null;
		//Iterator over relation
		Iterator<Similar<SimpleState, ComposedState>> ior=relation.iterator();
		
		boolean found=false;
		//find the state
		while(ior.hasNext()){
			currentSimilar=ior.next();
			if(currentSimilar.target.equals(targetState)){
				found=true;
				break;
			}
		}
		if(found){
			return currentSimilar.services;
		}
		return new HashSet<ComposedState>();
	}

	public Set<State[]> getAllSimilarState() {
		if(relation==null)
			return null;
		else
		{
			Set<State[]> result=new HashSet<State[]>();
			Iterator<Similar<SimpleState,ComposedState>> itRelation=relation.iterator();
			while(itRelation.hasNext()){
				Similar<SimpleState,ComposedState> currentSimilar=itRelation.next();
				SimpleState currentSimilarSimpleState=currentSimilar.target;
				Iterator<ComposedState> currentSimilarComposedStateIterator=currentSimilar.services.iterator();
				while(currentSimilarComposedStateIterator.hasNext())
					result.add(new State[]{currentSimilarSimpleState,
										   currentSimilarComposedStateIterator.next()
							              }
					          );
			}
			return result;
		}
	}
	
	
	@Override
	/**
	 * Returns the target Transition System
	 */
	public TransitionSystem<SimpleState, SimpleTransition> getTargetTs() {
		return target;
	}
	
	
	/**
	 * Generates the simulation of the target service with all the available services
	 * @param target the target service transition system
	 * @param availableServices the available services for the simulation
	 * @return a simulation of the target service
	 * @throws Exception if it's not possibile to simulate the service
	 */
	public static Simulation generate(	TransitionSystem<SimpleState, SimpleTransition> target,
										TransitionSystem<SimpleState, SimpleTransition>[] availableServices)
										throws Exception{
		
		
		//get the async-product result as a transition system
		TransitionSystem<ComposedState, ComposedTransition> async=asynchronousProduct(availableServices);
		//Iterator over target service states
		Iterator<SimpleState> iots=target.getAllStates().iterator();
		//Iterator over asynchronous product states
		Iterator<ComposedState> ioas;
		
		//holds target service state
		SimpleState tss;
		//holds similar state
		ComposedState ass;
		//holds the similar states
		HashSet<ComposedState> similarStates;
		//States which are similar
		Similar<SimpleState, ComposedState> similars;
		
		//result of the simulation
		SimulationStandardImpl result=new SimulationStandardImpl();
		result.relation=new HashSet<Similar<SimpleState,ComposedState>>();
		
		//-------------------------------------initial step everything is similar to everything---------------------
		// se lo stato da simulare � finale anche lo stato che simula deve essere finale. Modificato da Giovanni
		while(iots.hasNext()){
			tss=iots.next();
			similarStates=new HashSet<ComposedState>();
			//la riga sotto � stata aggiunga da giovanni
			ioas=async.getAllStates().iterator();
			while(ioas.hasNext()){
				ass=ioas.next();
				
				if(tss.isFinal() && ! ass.isFinal())
					continue;
				similarStates.add(ass);
			}
			
			//trick to use private class in static method
			similars=result.new Similar<SimpleState, ComposedState>();
			
			similars.target=tss;
			similars.services=similarStates;
			
			result.relation.add(similars);
		}
		
		//------------------------------------cutting non similar states----------------------------
		//iterator over relation
		Iterator<Similar<SimpleState, ComposedState>> ior=result.relation.iterator();
		
		//iterator over simple actions
		Iterator<SimpleTransition> iosa;
		//iterator over composed actions
		Iterator<ComposedTransition> ioca;
		//iterator over similar states
		Iterator<ComposedState> ioss;
		
		Transition currentAction;
		ComposedTransition simAction=null;//action that simulates the currrent one
		ComposedState currentComposed;
		//next set of similar states
		HashSet<ComposedState> nextSimilars;
		//current set of removed similars states
		HashSet<ComposedState> currentRemovedSimilars;
		
		boolean modified=true;//true if the relation has been modified
		boolean stateSimulated=true;//true if a certain state simulates another
		boolean actionSimulated=false;
		while(true==modified){
			
			modified=false;
			
			//if the iterator has no next element we start again to scan the relation
			if(!ior.hasNext()){
				ior=result.relation.iterator();
			}
			
			//analyze each state of the target service
			while(ior.hasNext()){
				similars=ior.next();//iterates over the relation {TS1, {SS11, SS21, SS31, ...}}
				nextSimilars=new HashSet<ComposedState>();
				currentRemovedSimilars=new HashSet<ComposedState>();
				ioss=similars.services.iterator();//Iterator over similar states (composed states)
				
				//analyze each state similar to the one of the target service
				while(ioss.hasNext()){
					stateSimulated=true;
					currentComposed=ioss.next();

					//actions for the current target service state
					iosa=target.getActionsOf(similars.target).iterator();

					//select each action of the target service
					while(iosa.hasNext()){
						//selection of an action
						currentAction=iosa.next();
						actionSimulated=false;
						
						ioca=async.getActionsOf(currentComposed).iterator();
						//can we simulate it?
						while(ioca.hasNext()){
							simAction=ioca.next();
							if(currentAction.actionSimulates(simAction)){
								//action exists
								if(!result.isSimulatedBy(currentAction.getStateTo(), simAction.getStateTo())){
									
									actionSimulated=false;
								}else{
										
									boolean selfEnding=simAction.getStateFrom().equals(simAction.getStateTo());
									//in the case of self ending actions checks the updates already done
									//to the relation between the current target state and the composed states
									if(selfEnding && currentRemovedSimilars.contains(simAction.getStateTo())){
										actionSimulated=false;
											
									}else{
										actionSimulated=true;
										break;
									}
								}
							}
						}
						if(!actionSimulated){
							stateSimulated=false;
							//if an action cannot be simulated, then the state is removed
							//so the state is not added to the new set of similar states
							//but instead we add it to the removed similar states
							currentRemovedSimilars.add(currentComposed);
							break;
						}else{
						}
					}
					if(stateSimulated){
						nextSimilars.add(currentComposed);
					}else{
						
						//we do not add the state to the new set, so the relation is modified
						modified=true;
					}
				}
				
				//a state has no similar composed states
				if(nextSimilars.isEmpty())
					throw new Exception("Unable to simulate service");
				
				similars.services=nextSimilars;
			}
		}

		
		
		//generates final composition
		result.target=target;
		result.composition=new TransitionSystemImpl<ComposedState, ComposedTransition>();
		ior=result.relation.iterator();

		
		//adds all states of the simulation
		while(ior.hasNext()){
			
			ioss=ior.next().services.iterator();
			while(ioss.hasNext()){
				try{
					result.composition.addState(ioss.next());
				}catch(DuplicateStateException e){
				}
			}
		}
		//adds all the actions
		ioss=result.composition.getAllStates().iterator();//iterates over the sets added to the final composition
		Iterator<ComposedState> ioss2=result.composition.getAllStates().iterator();//does the same as the one above
		while(ioss.hasNext()){
			currentComposed=ioss.next();//this is the "fixed" state
			//riga sotto aggiunta da giovanni
			ioss2=result.composition.getAllStates().iterator();
			while(ioss2.hasNext()){
				ioca=async.getActions(currentComposed, ioss2.next()).iterator();//gets all the actions connecting the "fixed" state to the ioss2.next() (another state in the composition)
				while(ioca.hasNext()){
					currentAction=ioca.next();
					result.composition.addAction((ComposedState)currentAction.getStateFrom(), (ComposedState)currentAction.getStateTo(), (ComposedTransition)currentAction);
				}
			}
		}
		
		
		//memorizes the asynchronous product
		result.async=async;

		return result;
	}

	/**
	 * Method that computes the asynchronous product between the given transition systems
	 * @param availableServices an array that contains the transition systems
	 * @return a composed transition system that represents the asynchronous product
	 * @throws Exception Error in the cartesian product have occurred
	 */
	@SuppressWarnings("unchecked")
	private static TransitionSystem<ComposedState, ComposedTransition> asynchronousProduct(TransitionSystem<SimpleState, SimpleTransition>[] availableServices) throws Exception {
		
		
		//-------------------------generates all the composed states by decapsulating the result of the recursive method----------
		
		//set of capsules
		HashSet<StateCapsule<SimpleState>> capsules=asynchronousProduct(availableServices, 0);
		
		//Set of composed states
		HashSet<ComposedState> states=new HashSet<ComposedState>();
		
		//iterator over capsules
		Iterator<StateCapsule<SimpleState>> it=capsules.iterator();
		StateCapsule currentCapsule;
		String composedStateName;
		State[] resultingStates=new State[availableServices.length];
		ServiceFactory sf=new ServiceFactoryImpl();
		while(it.hasNext()){
			currentCapsule=it.next();
			composedStateName="";
			int i=0;
			while(i<resultingStates.length){
				if(currentCapsule==null)
					throw new RuntimeException("Class: SimulationImpl Method: asynchronousProduct Exception: Problems during cartesian product have occurred");
				resultingStates[i]=currentCapsule.state;
				composedStateName+=currentCapsule.state.getName();
				currentCapsule=currentCapsule.sc;
				i++;
			}
			
			states.add(sf.createComposedState(composedStateName, resultingStates));
		}
		
		//-------------------------------------------actions analysis----------------------------------------------
		//2 iterators over the set of composed states
		Iterator<ComposedState> it1=states.iterator();
		//iterator over actions
		Iterator<? extends Transition> ioa;
		ComposedState fixed;//mantains the composed state from which the actions start
		ComposedState compared;//mantains the composed state to which an action ends
		Set<? extends Transition> currentActions;//temporary set to store actions of a certain state
		Set<ComposedTransition> actions=new HashSet<ComposedTransition>();
		SimpleState source;//source state of an action
		SimpleState end;//end state of an action
		Transition a;
		
		
		while(it1.hasNext()){
			fixed=it1.next();
			Iterator<ComposedState> it2=states.iterator();
			while(it2.hasNext()){
				compared=it2.next();
				int difference=fixed.compareNumberDifferentStates(compared);
				if(difference >1)
					continue;
				if(difference==-1)
					throw new Exception("Errors during the asynchronous product");
				for(int i=0; i<availableServices.length; i++){


					
					/*
					 * Composed states:
					 * FIXED:   {S10, S11, S12, S13, S14}
					 * COMPARED:{S20, S21, S22, S23, S24}
					 * 
					 * iterates over the lenght of each composed state
					 * and looks for connections between each related pair of states.
					 * E.g.: search actions between fixed[1] and compared[1]
					 */
					
					//may find connections between states
					source=(SimpleState)fixed.getServicesState(i);
					end=(SimpleState)compared.getServicesState(i);
					//check that the the source and the end of an action are the same,
					//then the difference between the compared composed states must be 0
					if(source.equals(end) && difference!=0)
						continue;
					currentActions=availableServices[i].getActions(source, end);
					ioa=currentActions.iterator();
					while(ioa.hasNext()){
						a=ioa.next();
						actions.add(sf.createComposedAction(a.getName(), availableServices[i], fixed, compared));
					}
				}
			}
		}
		
		//now generate the asynchronous product transition system
		Iterator<ComposedState> ics=states.iterator();
		Iterator<ComposedTransition> ica=actions.iterator();
		
		TransitionSystem<ComposedState, ComposedTransition> async=sf.createComposedTransitionSystem();
		
		while(ics.hasNext()){
			async.addState(ics.next());
		}
		
		while(ica.hasNext()){
			a=ica.next();
			async.addAction(((ComposedTransition)a).getStateFrom(), ((ComposedTransition)a).getStateTo(), (ComposedTransition)a);
			
		}
		
		
		return async;
	}
	
	
	
	/**
	 * Recursive method that makes cartesian product of the states over the last non ignored transition systems in the array.
	 * @param availableServices the transition systems of the services on which operate the cartesian product
	 * @param ignore the number of target services to ignore in the cartesian product
	 * @return a set containing encapsulated states.
	 */
	private static HashSet<StateCapsule<SimpleState>> asynchronousProduct(TransitionSystem<SimpleState, SimpleTransition>[] availableServices, int ignore){
		if(ignore>=availableServices.length-1){
			
			//this branch just encapsulates the states of the last transition system.
			HashSet<StateCapsule<SimpleState>> capsulesContainer=new HashSet<StateCapsule<SimpleState>>();
			
			Iterator<SimpleState> iss=availableServices[availableServices.length-1].getAllStates().iterator();
			SimpleState current;
			SimulationStandardImpl simpl=new SimulationStandardImpl();//dummy object to use Capsules
			StateCapsule<SimpleState> newCapsule;
			//creates new capsules for the very last part of each composed state
			//if a service has states s1, s2
			//then capsules will be:
			//{s1,null}
			//{s2,null}
			while(iss.hasNext()){
				current=iss.next();
				newCapsule=simpl.new StateCapsule<SimpleState>();
				newCapsule.state=current;
				newCapsule.sc=null;
				capsulesContainer.add(newCapsule);
			}
			return capsulesContainer;
		}else{
			
			//represent all the possible tail parts of a composed state
			HashSet<StateCapsule<SimpleState>> oldCapsules=asynchronousProduct(availableServices, ignore+1);
			
			//creates a new empty set of capsules
			HashSet<StateCapsule<SimpleState>> newCapsules=new HashSet<StateCapsule<SimpleState>>();
			
			
			//iterator over graph states
			Iterator<SimpleState> iogs=availableServices[ignore].getAllStates().iterator();
			
			
			
			
			
			
			//iterator over old capsules
			//old capsules represent the "tail" of each composed state
			//supposing the current service has states s1 s2
			//and old capsules contains states
			//
			//
			//q1r1 (represented as {q1,{r1,{null}}}
			//q1r2
			//q2r1
			//q2r2
			//Then new capsules will contain
			//s1q1r1
			//s1q1r2
			//s1q2r1
			//s1q2r2
			//s2q1r1
			//...
			Iterator<StateCapsule<SimpleState>> ioc;
			
			SimulationStandardImpl dummy=new SimulationStandardImpl();//dummy object to manage capsules
			
			SimpleState currentState;
			StateCapsule<SimpleState> currentCapsule;
			StateCapsule<SimpleState> newCapsule;
			while(iogs.hasNext()){
				currentState=iogs.next();
				ioc=oldCapsules.iterator();
				
				while(ioc.hasNext()){
					currentCapsule=ioc.next();
					
					newCapsule=dummy.new StateCapsule<SimpleState>();
					newCapsule.state=currentState;
					newCapsule.sc=currentCapsule;
					
					newCapsules.add(newCapsule);
				}
			}
			return newCapsules;
		}
	}
	
	/**
	 * Method that returns whether a certain state is simulated or not by any compound state
	 * @param state the state to check
	 * @param simulator the state that should simulate the given one
	 * @return <code>true</code> if the given state is simulated, <code>false</code> otherwise
	 */
	private boolean isSimulatedBy(State state, ComposedState simulator){
		Iterator<Similar<SimpleState, ComposedState>> it=relation.iterator();
		Similar<SimpleState, ComposedState> similars;
		while(it.hasNext()){
			similars=it.next();
			//the state of interest has been found
			if(similars.target.equals(state)){
				return similars.services.contains(simulator);
			}
		}
		//given state is not present in the simulation
		return false;
	}
	
	
	public TransitionSystem<ComposedState, ComposedTransition> getAsyncProduct(){
		return async;
	}
}
